/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <librevenge-stream/librevenge-stream.h>
#include <libetonyek/libetonyek.h>
#include <librvngabw/librvngabw.hxx>

#include "writerperfect_utils.hxx"
#include "ABWStringDocumentHandler.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "pages2abw"

using libetonyek::EtonyekDocument;

class ABWOutputFileHelper
{
public:
	ABWOutputFileHelper(const char *outFileName) : mOutFileName(outFileName) {}
	~ABWOutputFileHelper() {}

	bool convertTextDocument(librevenge::RVNGInputStream &input)
	{
		writerperfectrvngabw::ABWStringDocumentHandler contentHandler;
		librvngabw::ABWTextGenerator collector(&contentHandler);
		if (!EtonyekDocument::parse(&input, &collector))
			return false;
		if (mOutFileName)
		{
			FILE *fhandle = fopen(mOutFileName, "wb");
			if (!fhandle)
				return false;
			fprintf(fhandle, "%s\n", contentHandler.getData().cstr());
			fclose(fhandle);
		}
		else
			printf("%s\n", contentHandler.getData().cstr());
		return true;
	}

	static EtonyekDocument::Confidence isSupportedFormat(librevenge::RVNGInputStream &input, EtonyekDocument::Type &type)
	{
		return EtonyekDocument::isSupported(&input, &type);
	}
private:
	//! the ouput file name
	char const *mOutFileName;
};

int printUsage(char const *name)
{
	UsageHelper usage(name, "converts Apple Pages documents to AbiWord.");
	usage.addToDescription("If OUTPUT is missing, the result is printed to standard output.\n");
	usage.addStdoutOption();
	return usage.printUsage();
}

int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(TOOLNAME);

	char *szInputFile = 0;
	char *szOutFile = 0;
	bool stdOutput = false;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--stdout"))
			stdOutput = true;
		else if (!strcmp(argv[i], "--version"))
			return UsageHelper::printVersion(TOOLNAME);
		else if (!strcmp(argv[i], "--help"))
			return printUsage(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile)
		return printUsage(TOOLNAME);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	ABWOutputFileHelper helper(szOutFile);

	shared_ptr<librevenge::RVNGInputStream> input;
	if (librevenge::RVNGDirectoryStream::isDirectory(szInputFile))
		input.reset(new librevenge::RVNGDirectoryStream(szInputFile));
	else
		input.reset(new librevenge::RVNGFileStream(szInputFile));

	EtonyekDocument::Type type = EtonyekDocument::TYPE_UNKNOWN;
	const EtonyekDocument::Confidence confidence = helper.isSupportedFormat(*input, type);
	if ((EtonyekDocument::CONFIDENCE_NONE == confidence) || (EtonyekDocument::TYPE_PAGES != type))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Page Document.\n");
		return 1;
	}

	if (EtonyekDocument::CONFIDENCE_SUPPORTED_PART == confidence)
	{
		input.reset(librevenge::RVNGDirectoryStream::createForParent(szInputFile));

		if (EtonyekDocument::CONFIDENCE_EXCELLENT != helper.isSupportedFormat(*input, type) || (EtonyekDocument::TYPE_PAGES != type))
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Page Document.\n");
			return 1;
		}
	}

	if (!helper.convertTextDocument(*input))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
