/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file implements classes SKGError.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
*/
#include "skgerror.h"

SKGError::SKGError()
    : QObject(), m_rc(0), m_message(""), m_previousError(NULL)
{}

SKGError::SKGError(int iRc, const QString& iMessage)
    : QObject(), m_rc(iRc), m_message(iMessage), m_previousError(NULL)
{}

SKGError::SKGError(const SKGError& iError)
    : QObject()
{
    m_rc = iError.m_rc;
    m_message = iError.m_message;
    if (Q_UNLIKELY(iError.m_previousError != NULL)) {
        m_previousError = new SKGError(*iError.m_previousError);
    } else {
        m_previousError = NULL;
    }
}

SKGError::~SKGError()
{
    delete m_previousError;
    m_previousError = NULL;
}

const SKGError& SKGError::operator= (const SKGError& iError)
{
    if (Q_LIKELY(&iError != this)) {
        delete m_previousError;
        m_previousError = NULL;

        m_rc = iError.m_rc;
        m_message = iError.m_message;
        if (Q_UNLIKELY(iError.m_previousError != NULL)) {
            m_previousError = new SKGError(*iError.m_previousError);
        }
    }
    return *this;
}

bool SKGError::operator!() const
{
    return isSucceeded();
}

SKGError::operator bool() const
{
    return isFailed();
}

bool SKGError::isFailed() const
{
    return (m_rc > 0);
}

bool SKGError::isSucceeded() const
{
    return (m_rc <= 0);
}

bool SKGError::isWarning() const
{
    return (m_rc < 0);
}

int SKGError::getReturnCode() const
{
    return m_rc;
}

SKGError& SKGError::setReturnCode(int iReturnCode)
{
    m_rc = iReturnCode;
    return *this;
}

SKGError& SKGError::setMessage(const QString& iMessage)
{
    m_message = iMessage;
    return *this;
}

QString SKGError::getMessage() const
{
    return m_message;
}

QString SKGError::getFullMessage() const
{
    QString output('[');
    output += (m_rc == 0 ? "SUC" : (m_rc > 0 ? "ERR" : "WAR"));
    output += '-';

    QString tmp;
    tmp.setNum(m_rc);
    output += tmp;
    output += ']';
    if (Q_LIKELY(!m_message.isEmpty())) {
        output += ": " % m_message;
    }
    return output;
}

QString SKGError::getFullMessageWithHistorical() const
{
    QString output = getFullMessage();
    if (Q_UNLIKELY(m_previousError)) {
        output += '\n' % m_previousError->getFullMessageWithHistorical();
    }
    return output;
}

int SKGError::getHistoricalSize() const
{
    int output = 0;
    if (Q_UNLIKELY(m_previousError)) {
        output += 1 + m_previousError->getHistoricalSize();
    }
    return output;
}

SKGError& SKGError::addError(int iRc, const QString& iMessage)
{
    SKGError* pe = new SKGError(*this);
    setReturnCode(iRc);
    setMessage(iMessage);
    delete m_previousError;
    m_previousError = pe;
    return *this;
}

SKGError& SKGError::addError(const SKGError& iError)
{
    return addError(iError.getReturnCode(), iError.getMessage());
}

SKGError* SKGError::getPreviousError() const
{
    return m_previousError;
}

#include "skgerror.moc"
