/***************************************************************************
 *   Copyright © 2011 Jonathan Thomas <echidnaman@kubuntu.org>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU General Public License as        *
 *   published by the Free Software Foundation; either version 2 of        *
 *   the License or (at your option) version 3 or any later version        *
 *   accepted by the membership of KDE e.V. (or its successor approved     *
 *   by the membership of KDE e.V.), which shall act as a proxy            *
 *   defined in Section 14 of version 3 of the license.                    *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 ***************************************************************************/

#include "ReviewWidget.h"

#include <QtCore/QStringBuilder>
#include <QHBoxLayout>
#include <QtWidgets/QLabel>

#include <KGlobal>
#include <KLocalizedString>
#include <Nepomuk/KRatingWidget>

#include <ReviewsBackend/Review.h>
#include <resources/AbstractResource.h>

ReviewWidget::ReviewWidget(QWidget *parent)
        : KVBox(parent)
{
    QWidget *headerWidget = new QWidget(this);
    QHBoxLayout *headerLayout = new QHBoxLayout(headerWidget);
    headerLayout->setMargin(0);
    headerWidget->setLayout(headerLayout);

    m_ratingWidget = new KRatingWidget(headerWidget);
    m_ratingWidget->setAttribute(Qt::WA_TransparentForMouseEvents);
    m_ratingWidget->setPixmapSize(16);
    m_summaryLabel = new QLabel(headerWidget);
    QWidget *headerSpacer = new QWidget(headerWidget);
    headerSpacer->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    m_nameDateLabel = new QLabel(headerWidget);
    m_nameDateLabel->setEnabled(false);

    headerLayout->addWidget(m_ratingWidget);
    headerLayout->addWidget(m_summaryLabel);
    headerLayout->addWidget(headerSpacer);
    headerLayout->addWidget(m_nameDateLabel);

    m_reviewLabel = new QLabel(this);
    m_reviewLabel->setWordWrap(true);

    m_versionLabel = new QLabel(this);
    m_versionLabel->setEnabled(false);
    m_versionLabel->hide();

    QWidget *usefulnessWidget = new QWidget(this);
    QHBoxLayout *usefulnessLayout = new QHBoxLayout(usefulnessWidget);
    usefulnessLayout->setMargin(0);
    usefulnessWidget->setLayout(usefulnessLayout);

    m_usefulnessLabel = new QLabel(usefulnessWidget);
    // TODO: Report usefulness/inappropriateness once submitting works

    usefulnessLayout->addWidget(m_usefulnessLabel);
}

ReviewWidget::~ReviewWidget()
{
}

static int compareVersion(const QString& a, const QString& b)
{
    if(a==b)
        return 0;
    QStringList verA = a.split('.');
    QStringList verB = b.split('.');
    while(!verA.isEmpty() && !verB.isEmpty()) {
        QString curA = verA.takeFirst(), curB = verB.takeFirst();
        int diff = curB.compare(curA);
        if(diff!=0)
            return diff;
    }
    return verB.size()-verA.size();
}

static QString upstreamVersion(const QString& ver)
{
    QString ret = ver;
    
    int idx = ret.indexOf(':');
    if(idx>=0)
        ret= ret.mid(idx+1);
    
    idx = ret.indexOf('-');
    if(idx>=0)
        ret= ret.left(idx);
    
    idx = ret.indexOf('+');
    if(idx>=0)
        ret= ret.left(idx);
    
    return ret;
}

void ReviewWidget::setReview(Review *review)
{
    if (!review->package())
        return;

    m_ratingWidget->setRating(review->rating());

    m_summaryLabel->setText(QLatin1Literal("<b>") % review->summary()
                            % QLatin1Literal("</b>"));

    QString date = KGlobal::locale()->formatDate(review->creationDate().date(), KLocale::ShortDate);
    m_nameDateLabel->setText(i18nc("@label Formatted: username, date",
                                   "%1, %2",
                                   review->reviewer(), date));

    m_reviewLabel->setText(review->reviewText());

    const QString reviewUpstream = upstreamVersion(review->packageVersion());
    const QString currentUpstream = upstreamVersion(review->package()->installedVersion());

    int res = compareVersion(reviewUpstream, currentUpstream);

    if (res < 0) {
        m_versionLabel->setText(QLatin1Literal("<i>") %
                                i18nc("@label", "This review was written for an older version "
                                                "(Version: %1)", reviewUpstream) %
                                QLatin1Literal("</i>"));
        m_versionLabel->show();
    }

    if (review->usefulnessTotal()) {
        m_usefulnessLabel->setText(i18ncp("@label", "%1 out of %2 person found this review useful",
                                          "%1 out of %2 people found this review useful",
                                          review->usefulnessFavorable(), review->usefulnessTotal()));
    }
}

#include "ReviewWidget.moc"
