/******************************************************************************
 * This file is part of the libqgit2 library
 * Copyright (c) 2011 Laszlo Papp <djszapi@archlinux.us>
 * Copyright (C) 2013 Leonardo Giordani
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef LIBQGIT2_SIGNATURE_H
#define LIBQGIT2_SIGNATURE_H

#include <QtCore/QString>
#include <QtCore/QDateTime>
#include <QtCore/QSharedPointer>

#include "git2.h"

#include "libqgit2_config.h"

namespace LibQGit2
{
    /**
     * @brief This class represents signatures, used e.g. when creating commits.
     *
     * @ingroup LibQGit2
     * @{
     */
    class LIBQGIT2_EXPORT Signature
    {
        public:
            /**
             * Construct a new action signature.
             *
             * @param name name of the person
             * @param email email of the person
             * @param dateTime time when the action happened
             * @throws LibQGit2::Exception
             */
            Signature(const QString& name, const QString& email, QDateTime dateTime);

            /**
             * Construct a new action signature, using the `now` datetime generated by the
             * underlaying library.
             *
             * @param name name of the person
             * @param email email of the person
             * @throws LibQGit2::Exception
             */
            Signature(const QString& name, const QString& email);

            /**
             * Create a reference to an existing \a signature; ownership of the pointer is not transferred.
             */
            explicit Signature(const git_signature *signature = 0);

            /**
             * Return the 'name' from this signature
             */
            QString name() const;

            /**
             * Return the 'email' from this signature
             */
            QString email() const;

            /**
             * Return the time stamp from this signature
             */
            QDateTime when() const;

            const git_signature *data() const;

        private:
            class Private;
            QSharedPointer<Private> d_ptr;
    };

    /**@}*/
}

#endif // LIBQGIT2_SIGNATURE_H
