
/**
 *
 *  This file is part of the kdewebdev package
 *  Copyright (c) 2006 Keith Isdale <keith@kdewebdev.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include <QTableWidget> 
#include <QLineEdit> 
#include <QLabel>
#include <QPushButton>
#include <QTimer>

#include "xsldbgvariablesimpl.h" 
#include "xsldbgdebugger.h"
#include <klocale.h> 



XsldbgVariablesImpl::XsldbgVariablesImpl(XsldbgDebugger *debugger,
        QWidget *parent /*=0*/) 
:  QWidget(parent)
{	 
    setupUi(this);
    this->debugger = debugger; 
    connect(debugger, SIGNAL(variableItem(QString /*name */, QString /* templateContext*/,
                    QString /* fileName */, int /*lineNumber */, 
                    QString /* select XPath */, bool /* is it a local variable */)),	
            this, SLOT(slotProcVariableItem(QString /*name */, QString /* templateContext*/,
                    QString /* fileName */, int /*lineNumber */, 
                    QString /* select XPath */, bool /* is it a local variable */)));	
    connect(varsView,  SIGNAL(itemSelectionChanged()), 
            this, SLOT(itemSelectionChanged())); 

    connect(&resizeTimer, SIGNAL(timeout()), this, SLOT(resizeContents()));
    resizeContents();

    show(); 
} 


XsldbgVariablesImpl::~XsldbgVariablesImpl() 
{ 
    debugger = 0L; 
} 

void XsldbgVariablesImpl::slotProcVariableItem(QString name , QString templateContext,
        QString fileName, int lineNumber, 
        QString selectXPath, bool localVariable)
{ 

    if (!name.isNull()){
        resizeTimer.stop();
        if (insertPosition >= varsView->rowCount())
            varsView->insertRow(insertPosition);
        QTableWidgetItem *cellItem;
        for (int column=NameColumn; column<=LineNumberColumn; column++){
            cellItem = varsView->item(insertPosition, column); 
            if (!cellItem)
                varsView->setItem(insertPosition, column, new QTableWidgetItem);
            cellItem = varsView->item(insertPosition, column); 
            if (cellItem){
                switch(column){
                    case NameColumn:
                        cellItem->setText(name);
                        break;
                    case TemplateColumn:
                        cellItem->setText(templateContext);
                        break;
                    case TypeColumn:
                        if (localVariable)
                            cellItem->setText(i18n("Local"));
                        else
                            cellItem->setText(i18n("Global"));
                        break;
                    case XPathColumn:
                        cellItem->setText(selectXPath);
                        break;
                    case FileNameColumn:
                        cellItem->setText(fileName);
                        break;
                    case LineNumberColumn:
                        if (lineNumber != -1)
                            cellItem->setText(QString::number(lineNumber));
                        else
                            cellItem->setText("");
                        break;
                }
            }
        }
        resizeTimer.start();
        insertPosition++;
    }
}	 

void XsldbgVariablesImpl::itemSelectionChanged() 
{ 
    QList<QTableWidgetItem *> items = varsView->selectedItems();
    if (!items.isEmpty()){
        QString fileName;
        int lineNumber = 0;
        bool ok=false;
        QTableWidgetItem *cellItem;
        for (int i = 0; i < items.size(); ++i) {
            cellItem = items[i];	
            switch(cellItem->column()){
                case FileNameColumn:
                    fileName = cellItem->text();
                    break;
                case LineNumberColumn:
                    lineNumber = cellItem->text().toInt(&ok);
                    if (!ok)
                        lineNumber = 0;
                    break;
                case NameColumn:
                    variableName->setText(cellItem->text());
                    break;
                case TypeColumn:
                    variableType->setText(cellItem->text());
                    break;
                case XPathColumn:
                    xPathEdit->setText(cellItem->text());
                    break;
            }
        }
        // do not allow the variable to be edited if it does not have a value already
        setExpressionButton->setEnabled(!xPathEdit->text().isEmpty());
        xPathEdit->setEnabled(!xPathEdit->text().isEmpty());
        if (!fileName.isEmpty())
            debugger->gotoLine(fileName, lineNumber);	 
    }else{
        // "clear" values in variable editing widgets
        variableName->setText("");
        xPathEdit->setText("");
        variableType->setText("");
        setExpressionButton->setEnabled(false);
        xPathEdit->setEnabled(false);
    } 

} 

void XsldbgVariablesImpl::resizeContents()
{
    varsView->resizeColumnsToContents();
}

void XsldbgVariablesImpl::refresh() 
{
    if (varsView){
        varsView->clearContents(); 
        insertPosition = 0;
        // "clear" values in variable editing widgets
        variableName->setText("");
        xPathEdit->setText("");
        variableType->setText("");
        setExpressionButton->setEnabled(false);
        xPathEdit->setEnabled(false);
        debugger->fakeInput("locals -q", true) ; 
    }
} 

void XsldbgVariablesImpl::slotEvaluate()
{
    if (debugger != 0L)
        debugger->slotCatCmd( expressionEdit->text() );
}

void XsldbgVariablesImpl::slotSetExpression()
{
    if (debugger != 0L){
        debugger->slotSetVariableCmd( variableName->text(), xPathEdit->text() );
        refresh();
    }
}





#include "xsldbgvariablesimpl.moc"
