/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.util.crypto;

import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.security.util.crypto.AbstractSecureHasher;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.digests.MD5Digest;
import org.bouncycastle.crypto.digests.SHA1Digest;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.digests.SHA384Digest;
import org.bouncycastle.crypto.digests.SHA512Digest;
import org.bouncycastle.crypto.generators.PKCS5S2ParametersGenerator;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PBKDF2SecureHasher
extends AbstractSecureHasher {
    private static final Logger logger = LoggerFactory.getLogger(PBKDF2SecureHasher.class);
    private static final String DEFAULT_PRF = "SHA-512";
    private static final int DEFAULT_SALT_LENGTH = 16;
    private static final int DEFAULT_ITERATION_COUNT = 160000;
    private static final int DEFAULT_DK_LENGTH = 32;
    private static final int MIN_ITERATION_COUNT = 1;
    private static final int MIN_DK_LENGTH = 1;
    private static final int MIN_SALT_LENGTH = 8;
    private final Digest prf;
    private final Integer iterationCount;
    private final int dkLength;

    public PBKDF2SecureHasher() {
        this(DEFAULT_PRF, 160000, 0, 32);
    }

    public PBKDF2SecureHasher(int dkLength) {
        this(DEFAULT_PRF, 160000, 0, dkLength);
    }

    public PBKDF2SecureHasher(int iterationCount, int dkLength) {
        this(DEFAULT_PRF, iterationCount, 0, dkLength);
    }

    public PBKDF2SecureHasher(String prf, Integer iterationCount, int saltLength, int dkLength) {
        this.validateParameters(prf, iterationCount, saltLength, dkLength);
        this.prf = this.resolvePRF(prf);
        this.iterationCount = iterationCount;
        this.saltLength = saltLength;
        this.dkLength = dkLength;
    }

    private void validateParameters(String prf, Integer iterationCount, int saltLength, int dkLength) {
        logger.debug("Validating PBKDF2 secure hasher with prf {}, iteration count {}, salt length {} bytes, output length {} bytes", new Object[]{prf, iterationCount, saltLength, dkLength});
        if (!PBKDF2SecureHasher.isIterationCountValid(iterationCount)) {
            logger.error("The provided iteration count {} is below the minimum {}.", (Object)iterationCount, (Object)1);
            throw new IllegalArgumentException("Invalid iterationCount is not within iteration count boundary.");
        }
        this.initializeSalt(saltLength);
        Digest prfType = this.resolvePRF(prf);
        int hLen = prfType.getDigestSize();
        logger.debug("The PRF is {}, with a digest size (hLen) of {} bytes", (Object)prfType.getAlgorithmName(), (Object)hLen);
        if (!PBKDF2SecureHasher.isDKLengthValid(hLen, dkLength)) {
            logger.error("The provided dkLength {} bytes is outside the output boundary {} to {}.", new Object[]{dkLength, 1, PBKDF2SecureHasher.getMaxDKLength(hLen)});
            throw new IllegalArgumentException("Invalid dkLength is not within derived key length boundary.");
        }
    }

    @Override
    String getAlgorithmName() {
        return "PBKDF2";
    }

    @Override
    boolean acceptsEmptyInput() {
        return true;
    }

    public static boolean isIterationCountValid(Integer iterationCount) {
        if (iterationCount < 160000) {
            logger.warn("The provided iteration count {} is below the recommended minimum {}.", (Object)iterationCount, (Object)160000);
        }
        return iterationCount >= 1;
    }

    @Override
    int getDefaultSaltLength() {
        return 16;
    }

    @Override
    int getMinSaltLength() {
        return 8;
    }

    @Override
    int getMaxSaltLength() {
        return Integer.MAX_VALUE;
    }

    public static boolean isDKLengthValid(int hLen, Integer dkLength) {
        int MAX_DK_LENGTH = PBKDF2SecureHasher.getMaxDKLength(hLen);
        logger.debug("The max dkLength is {} bytes for hLen {} bytes.", (Object)MAX_DK_LENGTH, (Object)hLen);
        return dkLength >= 1 && dkLength <= MAX_DK_LENGTH;
    }

    private static int getMaxDKLength(int hLen) {
        long MAX_LENGTH = (Double.valueOf(Math.pow(2.0, 32.0)).longValue() - 1L) * (long)hLen;
        return Long.valueOf(Math.min(MAX_LENGTH, Integer.MAX_VALUE)).intValue();
    }

    @Override
    byte[] hash(byte[] input) {
        byte[] rawSalt = this.getSalt();
        return this.hash(input, rawSalt);
    }

    @Override
    byte[] hash(byte[] input, byte[] rawSalt) {
        logger.debug("Creating PBKDF2 hash with salt [{}] ({} bytes)", (Object)Hex.toHexString((byte[])rawSalt), (Object)rawSalt.length);
        if (!this.isSaltLengthValid(rawSalt.length)) {
            throw new IllegalArgumentException("The salt length (" + rawSalt.length + " bytes) is invalid");
        }
        long startNanos = System.nanoTime();
        PKCS5S2ParametersGenerator gen = new PKCS5S2ParametersGenerator(this.prf);
        gen.init(input, rawSalt, this.iterationCount.intValue());
        byte[] hash = ((KeyParameter)gen.generateDerivedParameters(this.dkLength * 8)).getKey();
        long generateNanos = System.nanoTime();
        long totalDurationMillis = TimeUnit.NANOSECONDS.toMillis(generateNanos - startNanos);
        logger.debug("Generated PBKDF2 hash in {} ms", (Object)totalDurationMillis);
        return hash;
    }

    private Digest resolvePRF(String prf) {
        if (StringUtils.isEmpty((CharSequence)prf)) {
            throw new IllegalArgumentException("Cannot resolve empty PRF");
        }
        String formattedPRF = prf.toLowerCase().replaceAll("[\\W]+", "");
        logger.debug("Resolved PRF {} to {}", (Object)prf, (Object)formattedPRF);
        switch (formattedPRF) {
            case "md5": {
                logger.warn("MD5 is a deprecated cryptographic hash function and should not be used");
                return new MD5Digest();
            }
            case "sha1": {
                logger.warn("SHA-1 is a deprecated cryptographic hash function and should not be used");
                return new SHA1Digest();
            }
            case "sha256": {
                return new SHA256Digest();
            }
            case "sha384": {
                return new SHA384Digest();
            }
            case "sha512": {
                return new SHA512Digest();
            }
        }
        logger.warn("Could not resolve PRF {}. Using default PRF {} instead", (Object)prf, (Object)DEFAULT_PRF);
        return new SHA512Digest();
    }
}

