/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.internal.core.ssl;

import com.datastax.oss.driver.shaded.guava.common.annotations.VisibleForTesting;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.Socket;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.Arrays;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.KeyManagerFactorySpi;
import javax.net.ssl.ManagerFactoryParameters;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.X509ExtendedKeyManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReloadingKeyManagerFactory
extends KeyManagerFactory
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ReloadingKeyManagerFactory.class);
    private static final String KEYSTORE_TYPE = "JKS";
    private Path keystorePath;
    private String keystorePassword;
    private ScheduledExecutorService executor;
    private final Spi spi;
    private volatile byte[] lastDigest;

    static ReloadingKeyManagerFactory create(Path keystorePath, String keystorePassword, Optional<Duration> reloadInterval) throws UnrecoverableKeyException, KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException {
        KeyStore ks;
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        try (InputStream ksf = Files.newInputStream(keystorePath, new OpenOption[0]);){
            ks = KeyStore.getInstance(KEYSTORE_TYPE);
            ks.load(ksf, keystorePassword.toCharArray());
        }
        kmf.init(ks, keystorePassword.toCharArray());
        ReloadingKeyManagerFactory reloadingKeyManagerFactory = new ReloadingKeyManagerFactory(kmf);
        reloadingKeyManagerFactory.start(keystorePath, keystorePassword, reloadInterval);
        return reloadingKeyManagerFactory;
    }

    @VisibleForTesting
    protected ReloadingKeyManagerFactory(KeyManagerFactory initial) {
        this(new Spi((X509ExtendedKeyManager)initial.getKeyManagers()[0]), initial.getProvider(), initial.getAlgorithm());
    }

    private ReloadingKeyManagerFactory(Spi spi, Provider provider, String algorithm) {
        super(spi, provider, algorithm);
        this.spi = spi;
    }

    private void start(Path keystorePath, String keystorePassword, Optional<Duration> reloadInterval) {
        this.keystorePath = keystorePath;
        this.keystorePassword = keystorePassword;
        this.reload();
        if (!reloadInterval.isPresent() || reloadInterval.get().isZero()) {
            String msg = "KeyStore reloading is disabled. If your Cassandra cluster requires client certificates, client application restarts are infrequent, and client certificates have short lifetimes, then your client may fail to re-establish connections to Cassandra hosts. To enable KeyStore reloading, see `advanced.ssl-engine-factory.keystore-reload-interval` in reference.conf.";
            logger.info("KeyStore reloading is disabled. If your Cassandra cluster requires client certificates, client application restarts are infrequent, and client certificates have short lifetimes, then your client may fail to re-establish connections to Cassandra hosts. To enable KeyStore reloading, see `advanced.ssl-engine-factory.keystore-reload-interval` in reference.conf.");
        } else {
            logger.info("KeyStore reloading is enabled with interval {}", (Object)reloadInterval.get());
            this.executor = Executors.newScheduledThreadPool(1, runnable -> {
                Thread t = Executors.defaultThreadFactory().newThread(runnable);
                t.setName(String.format("%s-%%d", this.getClass().getSimpleName()));
                t.setDaemon(true);
                return t;
            });
            this.executor.scheduleWithFixedDelay(this::reload, reloadInterval.get().toMillis(), reloadInterval.get().toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    @VisibleForTesting
    void reload() {
        try {
            this.reload0();
        }
        catch (Exception e) {
            String msg = "Failed to reload KeyStore. If this continues to happen, your client may use stale identity certificates and fail to re-establish connections to Cassandra hosts.";
            logger.warn(msg, (Throwable)e);
        }
    }

    private synchronized void reload0() throws NoSuchAlgorithmException, IOException, KeyStoreException, CertificateException, UnrecoverableKeyException {
        logger.debug("Checking KeyStore file {} for updates", (Object)this.keystorePath);
        byte[] keyStoreBytes = Files.readAllBytes(this.keystorePath);
        byte[] newDigest = ReloadingKeyManagerFactory.digest(keyStoreBytes);
        if (this.lastDigest != null && Arrays.equals(this.lastDigest, ReloadingKeyManagerFactory.digest(keyStoreBytes))) {
            logger.debug("KeyStore file content has not changed; skipping update");
            return;
        }
        KeyStore keyStore = KeyStore.getInstance(KEYSTORE_TYPE);
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(keyStoreBytes);){
            keyStore.load(inputStream, this.keystorePassword.toCharArray());
        }
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        kmf.init(keyStore, this.keystorePassword.toCharArray());
        logger.info("Detected updates to KeyStore file {}", (Object)this.keystorePath);
        this.spi.keyManager.set((X509ExtendedKeyManager)kmf.getKeyManagers()[0]);
        this.lastDigest = newDigest;
    }

    @Override
    public void close() throws Exception {
        if (this.executor != null) {
            this.executor.shutdown();
        }
    }

    private static byte[] digest(byte[] payload) throws NoSuchAlgorithmException {
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        return digest.digest(payload);
    }

    private static class DelegatingKeyManager
    extends X509ExtendedKeyManager {
        AtomicReference<X509ExtendedKeyManager> delegate;

        DelegatingKeyManager(X509ExtendedKeyManager initial) {
            this.delegate = new AtomicReference<X509ExtendedKeyManager>(initial);
        }

        void set(X509ExtendedKeyManager keyManager) {
            this.delegate.set(keyManager);
        }

        @Override
        public String chooseEngineClientAlias(String[] keyType, Principal[] issuers, SSLEngine engine) {
            return this.delegate.get().chooseEngineClientAlias(keyType, issuers, engine);
        }

        @Override
        public String chooseEngineServerAlias(String keyType, Principal[] issuers, SSLEngine engine) {
            return this.delegate.get().chooseEngineServerAlias(keyType, issuers, engine);
        }

        @Override
        public String[] getClientAliases(String keyType, Principal[] issuers) {
            return this.delegate.get().getClientAliases(keyType, issuers);
        }

        @Override
        public String chooseClientAlias(String[] keyType, Principal[] issuers, Socket socket) {
            return this.delegate.get().chooseClientAlias(keyType, issuers, socket);
        }

        @Override
        public String[] getServerAliases(String keyType, Principal[] issuers) {
            return this.delegate.get().getServerAliases(keyType, issuers);
        }

        @Override
        public String chooseServerAlias(String keyType, Principal[] issuers, Socket socket) {
            return this.delegate.get().chooseServerAlias(keyType, issuers, socket);
        }

        @Override
        public X509Certificate[] getCertificateChain(String alias) {
            return this.delegate.get().getCertificateChain(alias);
        }

        @Override
        public PrivateKey getPrivateKey(String alias) {
            return this.delegate.get().getPrivateKey(alias);
        }
    }

    private static class Spi
    extends KeyManagerFactorySpi {
        DelegatingKeyManager keyManager;

        Spi(X509ExtendedKeyManager initial) {
            this.keyManager = new DelegatingKeyManager(initial);
        }

        @Override
        protected void engineInit(KeyStore ks, char[] password) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected void engineInit(ManagerFactoryParameters spec) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected KeyManager[] engineGetKeyManagers() {
            return new KeyManager[]{this.keyManager};
        }
    }
}

